<?php
namespace SahelCore\CPT\Shortcodes\ProductPair;

use SahelCore\Lib;

class ProductPair implements Lib\ShortcodeInterface {
	private $base;
	
	function __construct() {
		$this->base = 'eltdf_product_pair';
		
		add_action('vc_before_init', array($this,'vcMap'));
		
		//Small product id filter
		add_filter( 'vc_autocomplete_eltdf_product_pair_small_product_id_callback', array( &$this, 'smallProductIdAutocompleteSuggester', ), 10, 1 );
		
		//Small product id render
		add_filter( 'vc_autocomplete_eltdf_product_pair_small_product_id_render', array( &$this, 'smallProductIdAutocompleteRender', ), 10, 1 );

		//Big product id filter
		add_filter( 'vc_autocomplete_eltdf_product_pair_big_product_id_callback', array( &$this, 'bigProductIdAutocompleteSuggester', ), 10, 1 );

		//Big product id render
		add_filter( 'vc_autocomplete_eltdf_product_pair_big_product_id_render', array( &$this, 'bigProductIdAutocompleteRender', ), 10, 1 );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if(function_exists('vc_map')) {
			vc_map(
				array(
					'name'                      => esc_html__( 'Product Pair', 'sahel' ),
					'base'                      => $this->getBase(),
					'category'                  => esc_html__( 'by SAHEL', 'sahel' ),
					'icon'                      => 'icon-wpb-product-pair extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'params'                    => array(
						array(
							'type'        => 'autocomplete',
							'param_name'  => 'small_product_id',
							'heading'     => esc_html__( 'Small Product', 'sahel' ),
							'settings'    => array(
								'sortable'      => true,
								'unique_values' => true
							),
							'admin_label' => true,
							'save_always' => true,
							'description' => esc_html__( 'If you left this field empty then product ID will be of the current page', 'sahel' )
						),array(
                            'type'        => 'dropdown',
                            'param_name'  => 'enable_custom_image_1',
                            'heading'     => esc_html__( 'Enable Custom Image For Small Product', 'sahel' ),
                            'value'       => array_flip(sahel_elated_get_yes_no_select_array(false)),
                            'admin_label' => true,
                            'save_always' => true,
                        ),
                        array(
                            'type'        => 'attach_image',
                            'param_name'  => 'custom_image_1',
                            'heading'     => esc_html__( 'Custom Image', 'sahel' ),
                            'value'       => sahel_elated_get_yes_no_select_array(false),
                            'admin_label' => true,
                            'save_always' => true,
                            'dependency'  => array('element' => 'enable_custom_image_1', 'value' => 'yes')
                        ),
						array(
							'type'        => 'autocomplete',
							'param_name'  => 'big_product_id',
							'heading'     => esc_html__( 'Big Product', 'sahel' ),
							'settings'    => array(
								'sortable'      => true,
								'unique_values' => true
							),
							'admin_label' => true,
							'save_always' => true,
							'description' => esc_html__( 'If you left this field empty then product ID will be of the current page', 'sahel' )
						),
                        array(
                            'type'        => 'dropdown',
                            'param_name'  => 'enable_custom_image_2',
                            'heading'     => esc_html__( 'Enable Custom Image For Big Image', 'sahel' ),
                            'value'       => array_flip(sahel_elated_get_yes_no_select_array(false)),
                            'admin_label' => true,
                            'save_always' => true,
                        ),
                        array(
                            'type'        => 'attach_image',
                            'param_name'  => 'custom_image_2',
                            'heading'     => esc_html__( 'Custom Image', 'sahel' ),
                            'value'       => sahel_elated_get_yes_no_select_array(false),
                            'admin_label' => true,
                            'save_always' => true,
                            'dependency'  => array('element' => 'enable_custom_image_2', 'value' => 'yes')
                        ),
						array(
							'type' => 'dropdown',
							'param_name' => 'layout',
							'heading' => esc_html__('Layout order','sahel'),
							'value' => array(
								esc_html__('Small Product On The Left','sahel') => '',
								esc_html__('Big Product On The Left','sahel') => 'eltdf-big-product-first',
							)
						),
						array(
							'type' => 'dropdown',
							'param_name' => 'title_tag',
							'heading' => esc_html__('Title Tag','sahel'),
							'value' => array_flip(sahel_elated_get_title_tag(true))
						),
						array(
							'type' => 'colorpicker',
							'param_name' => 'trim_background_color',
							'heading' => esc_html__('Trim Background Color','sahel'),
						),
						array(
						    'type'        => 'dropdown',
						    'param_name'  => 'parallax_scroll',
						    'heading'     => esc_html__( 'Enable Parallax Scroll Effect', 'sahel' ),
						    'value'       => sahel_elated_get_yes_no_select_array(false, true),
						    'admin_label' => true,
						),
					)
				)
			);
		}
    }
	
    public function render($atts, $content = null) {
        $args = array(
	        'small_product_id'          => '',
			'big_product_id'            => '',
            'enable_custom_image_1'     => 'no',
            'custom_image_1'            => '',
            'enable_custom_image_2'     => 'no',
            'custom_image_2'            => '',
			'layout'					=> '',
			'title_tag'					=> 'h5',
			'trim_background_color'		=> '',
			'parallax_scroll'			=> 'yes',
        );

		$params = shortcode_atts($args, $atts);

	    $params['small_product_id'] = !empty($params['small_product_id']) ? $params['small_product_id'] : get_the_ID();
		$params['big_product_id'] = !empty($params['big_product_id']) ? $params['big_product_id'] : get_the_ID();

		$params['title_tag'] = !empty($params['title_tag']) ? $params['title_tag'] : $args['title_tag'];

		$params['products'] = array();
		$params['products'][] = array('class_name' => 'eltdf-small-product', 'product_id' => $params['small_product_id'], 'image_size' => 'sahel_elated_image_square', 'enable_custom_image' => $params['enable_custom_image_1'], 'custom_image' => $params['custom_image_1'], 'display_button' => 'yes' );
		$params['products'][] = array('class_name' => 'eltdf-big-product', 'product_id' => $params['big_product_id'], 'image_size' => 'sahel_elated_image_portrait', 'enable_custom_image' => $params['enable_custom_image_2'], 'custom_image' => $params['custom_image_2'], 'display_button' => 'yes' );


		$params['holder_classes'] = $this->getHolderClasses($params);
		$params['trim_styles'] = $this->getTrimStyles($params);
		$params['parallax_data_1'] = $this->getParallaxData1($params);
		$params['parallax_data_2'] = $this->getParallaxData2($params);

		$html = sahel_elated_get_woo_shortcode_module_template_part( 'templates/product-pair', 'product-pair', '', $params );

		return $html;
	}
	
	/**
	 * Return product info styles
	 *
	 * @param $params
	 *
	 * @return array
	 */
	private function getHolderClasses( $params ) {
		$holderClasses = array();
		
		if ( ! empty( $params['layout'] ) ) {
			$holderClasses[] = $params['layout'];
		}

		if ($params['parallax_scroll'] == 'yes') {
			$holderClasses[] = 'eltdf-parallax-scroll';
		}
		
		return implode( ' ', $holderClasses );
	}
	
	/**
	 * Generates product title html based on id
	 *
	 * @param $params
	 *
	 * @return html
	 */
	public function getItemTitleHtml( $params ) {
		$html               = '';
		$product_id         = $params['product_id'];
		$title              = get_the_title( $product_id );
		$title_tag          = $params['title_tag'];

		if ( ! empty( $title ) ) {
			$html = '<' . esc_attr( $title_tag ) . ' itemprop="name" class="eltdf-pi-title entry-title">';
				$html .= '<a itemprop="url" href="' . esc_url( get_the_permalink( $product_id ) ) . '">' . esc_html( $title ) . '</a>';
			$html .= '</' . esc_attr( $title_tag ) . '>';
		}
		
		return $html;
	}
	
	/**
	 * Generates product price html based on id
	 *
	 * @param $params
	 *
	 * @return html
	 */
	public function getItemPriceHtml( $params ) {
		$html               = '';
		$product_id         = $params['product_id'];
		$product            = wc_get_product( $product_id );
		$currency = '<div class="eltdf-currency">' . get_woocommerce_currency_symbol() . '</div>';
		
		if ($product->is_in_stock() ) {

			if( $product->is_on_sale() ) {
				$price_html = $currency . $product->get_sale_price($context = 'view');
			}
			else $price_html = $currency . $product->get_regular_price();

			$html = '<div class="eltdf-pi-price">' . $price_html . '</div>';
		}
		
		return $html;
	}
	
	/**
	 * Generates product add to cart html based on id
	 *
	 * @param $params
	 *
	 * @return html
	 */
	public function getItemExcerptHtml( $params ) {
		$html               = '';
		$product_id         = $params['product_id'];

		if ( $product_excerpt = get_the_excerpt($product_id) ) {
			$html = '<div class="eltdf-pi-excerpt">' . $product_excerpt . '</div>';
		}

		return $html;
	}

	public function getParallaxData1( $params ) {
		$parallaxData = array();

		if ($params['parallax_scroll'] === 'yes') {
		    $y_absolute = rand(-80, -100);
		    $smoothness = 20; //1 is for linear, non-animated parallax

		    $parallaxData['data-parallax']= '{&quot;y&quot;: '.$y_absolute.', &quot;smoothness&quot;: '.$smoothness.'}';
		}

		return $parallaxData;
	}

	public function getParallaxData2( $params ) {
		$parallaxData = array();

		if ($params['parallax_scroll'] === 'yes') {
		    $y_absolute = rand(-20, -60);
		    $smoothness = 20; //1 is for linear, non-animated parallax

		    $parallaxData['data-parallax']= '{&quot;y&quot;: '.$y_absolute.', &quot;smoothness&quot;: '.$smoothness.'}';
		}

		return $parallaxData;
	}

	/**
	 * Filter small product by ID or Title
	 *
	 * @param $query
	 *
	 * @return array
	 */
	public function smallProductIdAutocompleteSuggester( $query ) {
		global $wpdb;
		$product_id = (int) $query;
		$post_meta_infos = $wpdb->get_results( $wpdb->prepare( "SELECT ID AS id, post_title AS title
					FROM {$wpdb->posts} 
					WHERE post_type = 'product' AND ( ID = '%d' OR post_title LIKE '%%%s%%' )", $product_id > 0 ? $product_id : - 1, stripslashes( $query ), stripslashes( $query ) ), ARRAY_A );

		$results = array();
		if ( is_array( $post_meta_infos ) && ! empty( $post_meta_infos ) ) {
			foreach ( $post_meta_infos as $value ) {
				$data = array();
				$data['value'] = $value['id'];
				$data['label'] = esc_html__( 'Id', 'sahel' ) . ': ' . $value['id'] . ( ( strlen( $value['title'] ) > 0 ) ? ' - ' . esc_html__( 'Title', 'sahel' ) . ': ' . $value['title'] : '' );
				$results[] = $data;
			}
		}

		return $results;

	}

	/**
	 * Find small product by id
	 * @since 4.4
	 *
	 * @param $query
	 *
	 * @return bool|array
	 */
	public function smallProductIdAutocompleteRender( $query ) {
		$query = trim( $query['value'] ); // get value from requested
		if ( ! empty( $query ) ) {
			
			$product = get_post( (int) $query );
			if ( ! is_wp_error( $product ) ) {
				
				$product_id = $product->ID;
				$product_title = $product->post_title;
				
				$product_title_display = '';
				if ( ! empty( $portfolio_title ) ) {
					$product_title_display = ' - ' . esc_html__( 'Title', 'sahel' ) . ': ' . $product_title;
				}
				
				$product_id_display = esc_html__( 'Id', 'sahel' ) . ': ' . $product_id;

				$data          = array();
				$data['value'] = $product_id;
				$data['label'] = $product_id_display . $product_title_display;

				return ! empty( $data ) ? $data : false;
			}

			return false;
		}

		return false;
	}

	/**
	 * Filter big product by ID or Title
	 *
	 * @param $query
	 *
	 * @return array
	 */
	public function bigProductIdAutocompleteSuggester( $query ) {
		global $wpdb;
		$product_id = (int) $query;
		$post_meta_infos = $wpdb->get_results( $wpdb->prepare( "SELECT ID AS id, post_title AS title
					FROM {$wpdb->posts}
					WHERE post_type = 'product' AND ( ID = '%d' OR post_title LIKE '%%%s%%' )", $product_id > 0 ? $product_id : - 1, stripslashes( $query ), stripslashes( $query ) ), ARRAY_A );

		$results = array();
		if ( is_array( $post_meta_infos ) && ! empty( $post_meta_infos ) ) {
			foreach ( $post_meta_infos as $value ) {
				$data = array();
				$data['value'] = $value['id'];
				$data['label'] = esc_html__( 'Id', 'sahel' ) . ': ' . $value['id'] . ( ( strlen( $value['title'] ) > 0 ) ? ' - ' . esc_html__( 'Title', 'sahel' ) . ': ' . $value['title'] : '' );
				$results[] = $data;
			}
		}

		return $results;

	}

	/**
	 * Find big product by id
	 * @since 4.4
	 *
	 * @param $query
	 *
	 * @return bool|array
	 */
	public function bigProductIdAutocompleteRender( $query ) {
		$query = trim( $query['value'] ); // get value from requested
		if ( ! empty( $query ) ) {

			$product = get_post( (int) $query );
			if ( ! is_wp_error( $product ) ) {

				$product_id = $product->ID;
				$product_title = $product->post_title;

				$product_title_display = '';
				if ( ! empty( $portfolio_title ) ) {
					$product_title_display = ' - ' . esc_html__( 'Title', 'sahel' ) . ': ' . $product_title;
				}

				$product_id_display = esc_html__( 'Id', 'sahel' ) . ': ' . $product_id;

				$data          = array();
				$data['value'] = $product_id;
				$data['label'] = $product_id_display . $product_title_display;

				return ! empty( $data ) ? $data : false;
			}

			return false;
		}

		return false;
	}


	public function getTrimStyles( $params ) {
		$styles = array();

		if (!empty($params['trim_background_color'])) {
			$styles[] = 'border-color: ' . $params['trim_background_color'] . ';';
		}

		return implode( ';', $styles );
	}
}